#region References

using System;
using System.Data;
using System.Data.SqlClient;
using System.Collections;
using System.Diagnostics;
using gov.va.med.vbecs.Common.Log;
using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages;
using gov.va.med.vbecs.Common;

#endregion

namespace gov.va.med.vbecs.DAL.HL7AL
{
	#region Header

	//<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	//<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	//<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	//<Developers>
	//	<Developer>Brian    lin</Developer>
	//</Developers>
	//<SiteName>Hines OIFO</SiteName>
	//<CreationDate>4/6/2004</CreationDate>
	//<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	//<summary></summary>

	#endregion

	/// <summary>
	/// VbecsHL7Bridge
	/// Renamed and modified slightly for BCE (CR 2961)
	/// </summary>
	public class VbecsHL7Bridge
	{
		#region Variables

		private const string LAST_UPDATE_USER_PARAMETER_NAME = "@LastUpdateUser";
		private const string LAST_UPDATE_USER_PARAMETER_VALUE = "CPRS";
		private const string LAST_UPDATE_FUNSTEOR_VARAMETER_NAME = "@LastUpdateFunctionId";
        // Events Logger
        private static readonly ILogger _eventsLogger =
            LogManager.Instance().LoggerLocator.GetLogger("SystemEvents");

		#endregion

		#region Private Methods

		/// <summary>
		/// 
		/// </summary>
		private static SqlParameter MakeLastUpdateUserSqlParam()
		{
			SqlParameter _param = new SqlParameter( LAST_UPDATE_USER_PARAMETER_NAME, LAST_UPDATE_USER_PARAMETER_VALUE );
			_param.DbType = DbType.String;
			return _param;
		}

		/// <summary>
		/// 
		/// </summary>
		private static SqlParameter MakeLastUpdateFunctionSqlParam()
		{
			SqlParameter _param = new SqlParameter( LAST_UPDATE_FUNSTEOR_VARAMETER_NAME, Common.UpdateFunction.HL7CprsInterface );
			_param.DbType = DbType.Int32;
			return _param;

		}

		#endregion

		#region Public Methods

		///<Developers>
		///	<Developer>Brian    lin, David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/11/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7024"> 
		///		<ExpectedInput>Valid HL7ProtocolMessage for an outbound CPRS order update message</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7025"> 
		///		<ExpectedInput>Null HL7ProtocolMessage input parameter</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Public method used to deliver an outbound HL7ProtocolMessage
		/// CR 2967
		/// </summary>
		/// <param name="hl7Message"><see cref="HL7ProtocolMessage"/></param>
		/// <param name="hl7Interface"><see cref="HL7Interface"/></param>
		/// <param name="transmitCount"></param>
		public static void SendVbecsOutboundMessage( HL7ProtocolMessage hl7Message, HL7Interface hl7Interface, int transmitCount ) 
		{
			if ( hl7Message == null )
			{ 
				throw ( new ArgumentNullException( "hl7Message" ) );
			}
			try
			{
				var broker = new VbecsHL7Broker();
				//
				if( broker.Logon(hl7Interface) )
				{
					#region WriteOutBoundEventEntry

					if( hl7Interface.LogEvents )
					{
						string eventEntry = string.Concat( 
							"HL7 Message: \n ", 
							hl7Message.GetMessage(), 
							"\nOn Address\\Port: \n ", 
							hl7Interface.InterfaceIPAddress, 
							"\\",
							hl7Interface.InterfacePortNumber, 
							EventLogEntryType.Information );
						//
						var source = string.Concat( "Outbound message to ", hl7Interface.InterfaceName );
						//
                        // EventLogAppender
                        _eventsLogger.Info(string.Format("{0}: {1}", source, eventEntry));
						//EventLog.WriteEntry( source, eventEntry );
					}

					#endregion
					//
					var ackMessage = broker.ExecuteHL7Send( hl7Message );
					//
					if( hl7Interface.LogEvents )
					{
						#region WriteResponseEventEntry

						var source = string.Concat( "Response message from ", hl7Interface.InterfaceName );
						//
                        // EventLogAppender
                        _eventsLogger.Info(string.Format("{0}: {1}", source, ackMessage.GetMessage()));
                        //EventLog.WriteEntry( source, ackMessage.GetMessage(), EventLogEntryType.Information );

						#endregion
					}
					try
					{
						#region InsertAckMessageLog

						// Default to CPRS
						Common.UpdateFunction function = Common.UpdateFunction.HL7CprsInterface;
						//
						switch ( hl7Interface.InterfaceControlId )
						{
							case (int) Common.InterfaceName.BCE_COTS:
								function = Common.UpdateFunction.HL7BceInterface;
								break;
							default:
								break;
						}
						//
						HL7MessageLog.InsertMessageLog( 
							HL7MessageLog.GetMessageDataForMessageLogInsert( 
							ackMessage.GetMessage(), 
							MessageStatus.SuccessfullyCompleted, 
							hl7Interface, 
							function ) );

						#endregion
						//
						#region InsertAckMessageControlId

						HL7MessageLog.InsertAckMessageControlId( 
							hl7Message.GetMessageControlID(), 
							ackMessage.GetMessageControlID(), 
							MessageStatus.SuccessfullyCompleted,
							hl7Interface.InterfaceName, 
							Common.UpdateFunction.VbecsHL7ClientMonitor );

						#endregion
					}
					catch( Exception exc )
					{
						#region WriteExceptionEventEntry

						var source = string.Concat( "VBECS HL7 Client Monitor, Interface Name: ", hl7Interface.InterfaceName );
						//
                        // EventLogAppender
                        // Carrie: look below in commented old code it was logged at FailureAudit level, log4net doesn't have similar level so I used Error level.
                        _eventsLogger.Error(string.Format("{0}: {1}", source, string.Concat("Error Inserting into MessageLog table: \n", exc.Message)));
						//EventLog.WriteEntry( source, 
						//	string.Concat("Error Inserting into MessageLog table: \n", exc.Message), 
						//	EventLogEntryType.FailureAudit );

						#endregion
					}
				}
				else
				{
					#region UpdateMessageStatus

					// If we can't logon, set the message status back to pending transmission.
					//
					HL7MessageLog.UpdateMessageStatus( 
						MessageStatus.PendingTransmission,
						hl7Message.GetMessageControlID(),
						null,
						// CR 2967
						(transmitCount + 1),
						"VBECS HL7 Client Monitor",
						UpdateFunction.VbecsHL7ClientMonitor );

					#endregion
				}
			}
			catch(HL7NetworkErrorException hl7exc)
			{
				#region UpdateMessageStatus

				// If we get a network error, set the status back to pending transmission.
				//
				HL7MessageLog.UpdateMessageStatus( 
					MessageStatus.PendingTransmission,
					hl7Message.GetMessageControlID(),
					hl7exc.Message,
					// CR 2967
					(transmitCount + 1),
					"VBECS HL7 Client Monitor",
					UpdateFunction.VbecsHL7ClientMonitor );

				#endregion
			}
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6066"> 
		///		<ExpectedInput>No input parameters required.</ExpectedInput>
		///		<ExpectedOutput>Integer greater than 0.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6067"> 
		///		<ExpectedInput>No Failure test available.</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets the next VBECS Order Number from the VbecsOrder table
		/// </summary>
		public static int GetNextVbecsOrderNumber()
		{
			SqlParameter[] _prms =
				{
					new SqlParameter( LAST_UPDATE_USER_PARAMETER_NAME, SqlDbType.VarChar),
					new SqlParameter( LAST_UPDATE_FUNSTEOR_VARAMETER_NAME, SqlDbType.Int),
			};

			_prms[0] = MakeLastUpdateUserSqlParam();
			_prms[1] = MakeLastUpdateFunctionSqlParam();
			
			DataTable dtOrderNumber = Common.StoredProcedure.GetData(STOREDPROC.HL7GetNextVbecsOrderNumber.StoredProcName, _prms).Tables[0];

			return Convert.ToInt32(dtOrderNumber.Rows[0]["VbecsOrderNumber"]);
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6064"> 
		///		<ExpectedInput>Valid HL7 message input parameter</ExpectedInput>
		///		<ExpectedOutput>Non-null HL7ProtocolMessage.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6065"> 
		///		<ExpectedInput>Null HL7 message input parameter</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Builds an HL7ProtocolMessage from a string
		/// </summary>
		/// <param name="hl7Message">HL7 Message string</param>
		/// <returns><see cref="HL7ProtocolMessage"/></returns>
		public static HL7ProtocolMessage BuildHL7ProtocolMessage( string hl7Message )
		{
			if( hl7Message == null )
			{
				throw( new ArgumentNullException( "hl7Message" ) );
			}
			//
			HL7ProtocolMessageFactory messageFactory = new HL7ProtocolMessageFactory();
			//
			return messageFactory.ParseCreateHL7ProtocolMessage( hl7Message );
		}

		#endregion
	}
}
